﻿using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.ComponentModel;
using System.Diagnostics;
using System.IO;
using System.Linq;
using System.Text;
using System.Windows;
using System.Windows.Controls;
using System.Windows.Data;
using System.Windows.Documents;
using System.Windows.Input;
using System.Windows.Media;
using System.Windows.Media.Imaging;
using System.Windows.Threading;

using Hosui.Debugger;
using Hosui.IO;
using Hosui.TextEditor;

namespace Hosui
{
    /// <summary>
    /// MainWindow.xaml の相互作用ロジック
    /// </summary>
    public partial class MainWindow : Window
    {
        public MainWindow()
        {
            InitializeComponent();
        }

        public HosuiContext Context
        {
            get { return (HosuiContext)GetValue(ContextProperty); }
            set { SetValue(ContextProperty, value); }
        }

        public static readonly DependencyProperty ContextProperty =
            DependencyProperty.Register("Context", typeof(HosuiContext), typeof(MainWindow));

        private void AsyncInvoke(Action action)
        {
            Dispatcher.BeginInvoke(action, DispatcherPriority.Background);
        }

        private void AsyncInvoke<T>(Action<T> action, T param)
        {
            Dispatcher.BeginInvoke(action, DispatcherPriority.Background, param);
        }

        private void MainWindow_Initialized(object sender, EventArgs e)
        {
            // ウィンドウの位置と状態を復元
            if (Settings.Current.Location.X != 0 || Settings.Current.Location.Y != 0)
            {
                Left = Settings.Current.Location.X;
                Top = Settings.Current.Location.Y;
            }
            if (Settings.Current.Size.Width != 0 || Settings.Current.Size.Height != 0)
            {
                Width = Settings.Current.Size.Width;
                Height = Settings.Current.Size.Height;
            }
            if (Settings.Current.SideWidth != 0)
            {
                sideColumn.Width = new GridLength(Settings.Current.SideWidth);
            }
            if (Settings.Current.MainRowHeight != 0)
            {
                mainRow.Height = new GridLength(Settings.Current.MainRowHeight);
            }
            if (Settings.Current.SideRowHeight != 0)
            {
                sideRow.Height = new GridLength(Settings.Current.SideRowHeight);
            }
            WindowState = Settings.Current.WindowState;
        }

        private void MainWindow_Loaded(object sender, RoutedEventArgs e)
        {
        }

        private void MainWindow_Closing(object sender, CancelEventArgs e)
        {
            if (!CheckModifiedFiles())
            {
                e.Cancel = true;
                return;
            }
            // ウィンドウの設定を保存する
            Settings.Current.Location = RestoreBounds.Location;
            Settings.Current.Size = RestoreBounds.Size;
            Settings.Current.WindowState = WindowState;
            Settings.Current.SideWidth = sideColumn.ActualWidth;
            Settings.Current.MainRowHeight = mainRow.ActualHeight;
            Settings.Current.SideRowHeight = sideRow.ActualHeight;
        }

        private void ExitMenuItem_Click(object sender, RoutedEventArgs e)
        {
            Close();
        }

        private void OptionMenuItem_Click(object sender, RoutedEventArgs e)
        {
        }

        private void EditorTabControl_SelectionChanged(object sender, SelectionChangedEventArgs e)
        {
            if (e.AddedItems.Count > 0)
            {
                var editingItem = (IEditableFile)e.AddedItems[0];
                editingItem.Editor.Focus();
            }
        }

        private void TaskListViewItem_MouseDoubleClick(object sender, MouseButtonEventArgs e)
        {
            var item = (ListViewItem)sender;
            var task = (Task)item.DataContext;
            var editingItem = Context.AddEditableFile(task.FullPath);
            if (editingItem == null)
            {
                return;
            }
            editorTabControl.SelectedItem = editingItem;
            editingItem.Editor.JumpTo(task.Line - 1);
            editingItem.Editor.Focus();
        }

        private void ExplorerTreeViewItem_MouseDoubleClick(object sender, MouseButtonEventArgs e)
        {
            var item = (TreeViewItem)sender;
            var entry = (FileEntry)item.DataContext;
            OpenFile(entry.FullPath);
        }

        private void OpenFile(string path)
        {
            var editingItem = Context.AddEditableFile(path);
            if (editingItem == null)
            {
                if (File.Exists(path))
                {
                    Process.Start(path);
                }
                return;
            }
            editorTabControl.SelectedItem = editingItem;
            editingItem.Editor.Focus();
        }

        private void EntryTreeViewItem_MouseDoubleClick(object sender, MouseButtonEventArgs e)
        {
            var item = (TreeViewItem)sender;
            var entry = (Entry)item.DataContext;
            if (entry.SubEntries.Count == 0)
            {
                var dialog = new EntryDialog { Entry = entry, Owner = this };
                if (dialog.ShowDialog() ?? false)
                {
                }
                e.Handled = true;
            }
        }

        private void FileCommand_CanExecute(object sender, CanExecuteRoutedEventArgs e)
        {
            e.CanExecute = Context != null;
        }

        private void NewCommand_Executed(object sender, ExecutedRoutedEventArgs e)
        {
            var editingItem = Context.AddDictionaryFile();
            editorTabControl.SelectedItem = editingItem;
            editingItem.Editor.Focus();
        }

        private void OpenCommand_Executed(object sender, ExecutedRoutedEventArgs e)
        {
            var dialog = new System.Windows.Forms.OpenFileDialog
            {
                AutoUpgradeEnabled = true,
                InitialDirectory = Context.RootDirectory,
                Filter = "華和梨辞書ファイル|*.kis|AYA 辞書ファイル|*.dic|すべてのファイル|*.*",
                FilterIndex = 0,
                Multiselect = false,
                CheckFileExists = true,
            };
            if (dialog.ShowDialog() == System.Windows.Forms.DialogResult.Cancel)
            {
                return;
            }
            var editingItem = Context.AddEditableFile(dialog.FileName);
            if (editingItem == null)
            {
                return;
            }
            editorTabControl.SelectedItem = editingItem;
            editingItem.Editor.Focus();
        }

        private void OpenProjectCommand_Executed(object sender, ExecutedRoutedEventArgs e)
        {
            var dialog = new System.Windows.Forms.FolderBrowserDialog
            {
                Description = "ゴーストディレクトリを指定してください",
                SelectedPath = Environment.GetFolderPath(Environment.SpecialFolder.MyDocuments)
            };
            if (dialog.ShowDialog() == System.Windows.Forms.DialogResult.Cancel)
            {
                return;
            }
            var directory = dialog.SelectedPath;
            OpenProject(directory);
        }

        private void OpenProject(string directory)
        {
            if (!File.Exists(Path.Combine(directory, @"ghost\master\descript.txt")))
            {
                return;
            }
            if (Settings.Current.RecentProjects.Contains(directory))
            {
                Settings.Current.RecentProjects.Move(Settings.Current.RecentProjects.IndexOf(directory), 0);
            }
            else
            {
                Settings.Current.RecentProjects.Insert(0, directory);
            }
            if (Settings.Current.RecentProjects.Count > 5)
            {
                Settings.Current.RecentProjects.RemoveAt(5);
            }
            Context = new HosuiContext(Path.Combine(directory, @"ghost\master"));
        }

        private void CloseCommand_Executed(object sender, ExecutedRoutedEventArgs e)
        {
            if (!CheckModifiedFiles())
            {
                return;
            }
            // 本来ならチェックが要る
            Context = null;
        }

        private bool CheckModifiedFile(IEditableFile item)
        {
            if (item.IsModified)
            {
                var result = System.Windows.Forms.MessageBox.Show(item.FileName + " は変更されています。保存しますか？", "豊水",
                    System.Windows.Forms.MessageBoxButtons.YesNoCancel, System.Windows.Forms.MessageBoxIcon.Warning, System.Windows.Forms.MessageBoxDefaultButton.Button1);
                switch (result)
                {
                    case System.Windows.Forms.DialogResult.Cancel:
                        return false;
                    case System.Windows.Forms.DialogResult.No:
                        break;
                    case System.Windows.Forms.DialogResult.Yes:
                        item.SaveFile();
                        break;
                }
            }
            return true;
        }

        private bool CheckModifiedFiles()
        {
            if (Context == null)
            {
                return true;
            }
            foreach (var item in Context.EditingFiles)
            {
                if (!CheckModifiedFile(item))
                {
                    return false;
                }
            }
            return true;
        }

        private void SaveCommand_Executed(object sender, ExecutedRoutedEventArgs e)
        {
            var item = (DictionaryFile)editorTabControl.SelectedItem;
            if (item.SaveFile())
            {
                Context.RefreshTaskList(item.FullPath);
            }
        }

        private void SaveAllCommand_Executed(object sender, ExecutedRoutedEventArgs e)
        {
            foreach (var item in Context.EditingFiles)
            {
                item.SaveFile();
            }
            Context.RefreshTaskList();
        }

        private void RecentProjectCommand_Executed(object sender, ExecutedRoutedEventArgs e)
        {
            var directory = (string)e.Parameter;
            OpenProject(directory);
        }

        private void CloseTabCommand_Executed(object sender, ExecutedRoutedEventArgs e)
        {
            var item = (DictionaryFile)editorTabControl.SelectedItem;
            if (CheckModifiedFile(item))
            {
                Context.EditingFiles.Remove(item);
            }
        }

        private void EditingCommand_CanExecute(object sender, CanExecuteRoutedEventArgs e)
        {
            e.CanExecute = editorTabControl != null && editorTabControl.SelectedItem != null;
        }

        private void CutCommand_Executed(object sender, ExecutedRoutedEventArgs e)
        {
            var item = (DictionaryFile)editorTabControl.SelectedItem;
            item.Editor.Cut();
        }

        private void CopyCommand_Executed(object sender, ExecutedRoutedEventArgs e)
        {
            var item = (DictionaryFile)editorTabControl.SelectedItem;
            item.Editor.Copy();
        }

        private void PasteCommand_Executed(object sender, ExecutedRoutedEventArgs e)
        {
            var item = (DictionaryFile)editorTabControl.SelectedItem;
            item.Editor.Paste();
        }

        private void DeleteCommand_Executed(object sender, ExecutedRoutedEventArgs e)
        {
            var item = (DictionaryFile)editorTabControl.SelectedItem;
            item.Editor.Delete();
        }

        private void SelectAllCommand_Executed(object sender, ExecutedRoutedEventArgs e)
        {
            var item = (DictionaryFile)editorTabControl.SelectedItem;
            item.Editor.SelectAll();
        }

        private void UndoCommand_CanExecute(object sender, CanExecuteRoutedEventArgs e)
        {
            e.CanExecute = editorTabControl != null && editorTabControl.SelectedItem != null ? ((DictionaryFile)editorTabControl.SelectedItem).Editor.TextEditor.EnableUndo : false;
        }

        private void UndoCommand_Executed(object sender, ExecutedRoutedEventArgs e)
        {
            var item = (DictionaryFile)editorTabControl.SelectedItem;
            item.Editor.Undo();
        }

        private void RedoCommand_CanExecute(object sender, CanExecuteRoutedEventArgs e)
        {
            e.CanExecute = editorTabControl != null && editorTabControl.SelectedItem != null ? ((DictionaryFile)editorTabControl.SelectedItem).Editor.TextEditor.EnableRedo : false;
        }

        private void RedoCommand_Executed(object sender, ExecutedRoutedEventArgs e)
        {
            var item = (DictionaryFile)editorTabControl.SelectedItem;
            item.Editor.Redo();
        }

        private void StartDebugCommand_Executed(object sender, ExecutedRoutedEventArgs e)
        {
            AsyncInvoke(() => Context.StartDebug());
        }

        private void StopDebugCommand_Executed(object sender, ExecutedRoutedEventArgs e)
        {
            commandOutput.Document.Blocks.Clear();
            AsyncInvoke(() => Context.StopDebug());
        }

        private void RestartDebugCommand_Executed(object sender, ExecutedRoutedEventArgs e)
        {
            commandOutput.Document.Blocks.Clear();
            AsyncInvoke(() => Context.RestartDebug());
        }

        private void PauseDebugCommand_Executed(object sender, ExecutedRoutedEventArgs e)
        {
        }

        private void AttachDebugCommand_Executed(object sender, ExecutedRoutedEventArgs e)
        {
            AsyncInvoke(() => Context.AttachDebug());
        }

        private void BeginDebuggerCommand_CanExecute(object sender, CanExecuteRoutedEventArgs e)
        {
            e.CanExecute = Context != null && Context.Module == null;
        }

        private void EndDebuggerCommand_CanExecute(object sender, CanExecuteRoutedEventArgs e)
        {
            e.CanExecute = Context != null && Context.Module != null;
        }

        private void ExplorerOpenMenuItem_Click(object sender, RoutedEventArgs e)
        {
            var item = (TreeViewItem)sender;
            var entry = (FileEntry)item.DataContext;
            if (File.Exists(entry.FullPath))
            {
                OpenFile(entry.FullPath);
            }
            else if (Directory.Exists(entry.FullPath))
            {
                Process.Start(entry.FullPath);
            }
        }

        private void ExplorerDeleteMenuItem_Click(object sender, RoutedEventArgs e)
        {
        }

        private void EntryAddMenuItem_Click(object sender, RoutedEventArgs e)
        {
        }

        private void EntryEditMenuItem_Click(object sender, RoutedEventArgs e)
        {
        }

        private void EntryDeleteMenuItem_Click(object sender, RoutedEventArgs e)
        {
            var item = (Entry)((FrameworkElement)sender).DataContext;
            if (item == null)
            {
                return;
            }
            var names = item.Name.Split('.');
            var entries = Context.Module.Entries;
            for (int i = 1; i < names.Length; i++)
            {
                var next = entries.FirstOrDefault(p => p.Name == names[i - 1]);
                if (next == null)
                {
                    return;
                }
                entries = next.SubEntries;
            }
            entries.Remove(item);
        }

        private void TextBox_KeyDown(object sender, KeyEventArgs e)
        {
            if (Context == null)
            {
                return;
            }
            if (e.Key != Key.Enter)
            {
                return;
            }
            var textBox = (TextBox)sender;
            if (textBox.Text.Length == 0)
            {
                return;
            }
            if (textBox.Text == ".")
            {
                textBox.Clear();
                Context.Mode = Context.Mode == CommandMode.Echo ? CommandMode.Command : CommandMode.Echo;
                return;
            }
            if (!Context.IsDebugging)
            {
                textBox.Clear();
                return;
            }
            if (Context.CommandHistory.IndexOf(textBox.Text) != -1)
            {
                Context.CommandHistory.Remove(textBox.Text);
            }
            Context.CommandHistory.Add(textBox.Text);
            Context.HistoryIndex = Context.CommandHistory.Count;
            var text = textBox.Text;
            var result = Context.Module.Evaluate(Context.Mode == CommandMode.Echo ? text : string.Format("$({0})", text));
            textBox.Clear();
            // 表示する
            var paragraph = new Paragraph { Margin = new Thickness(0) };
            paragraph.Inlines.Add(new Run(string.Format("{0} > {1}",
                Context.Mode == CommandMode.Echo ? "echo-mode" : "command-mode", text)) { Foreground = Brushes.Green });
            if (result.Length != 0)
            {
                paragraph.Inlines.Add(new LineBreak());
                paragraph.Inlines.Add(result);
            }
            commandOutput.Document.Blocks.Add(paragraph);
            commandOutput.ScrollToEnd();
            // 非同期でエントリを更新
            AsyncInvoke(Context.Module.Refresh);
        }

        private void TextBox_PreviewKeyDown(object sender, KeyEventArgs e)
        {
            if (Context == null)
            {
                return;
            }
            int index = Context.HistoryIndex;
            int count = Context.CommandHistory.Count;
            if (count == 0)
            {
                return;
            }
            var textBox = (TextBox)sender;
            if (e.Key == Key.Up)
            {
                if (index <= 0)
                {
                    index = count;
                    textBox.Text = string.Empty;
                }
                else
                {
                    --index;
                    textBox.Text = Context.CommandHistory[index];
                    textBox.SelectAll();
                    e.Handled = true;
                }
            }
            else if (e.Key == Key.Down)
            {
                ++index;
                if (index > count)
                {
                    index = 0;
                }
                else if (index == count)
                {
                    textBox.Text = string.Empty;
                    Context.HistoryIndex = index;
                    return;
                }
                textBox.Text = Context.CommandHistory[index];
                textBox.SelectAll();
                e.Handled = true;
            }
            Context.HistoryIndex = index;
        }
    }
}