﻿using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.ComponentModel;
using System.IO;
using System.Linq;
using System.Text;
using System.Windows;
using System.Windows.Data;

using Hosui.Debugger;
using Hosui.IO;
using Hosui.TextEditor;

namespace Hosui
{
    public class HosuiContext : PropertyChangedBase
    {
        public HosuiContext(string directory)
        {
            _directory = directory;
            Mode = CommandMode.Echo;
            // 定義ファイルを解析
            Utility.Serializer.Deserialize(Path.Combine(directory, "descript.txt"), _definitions);
            // ファイルリストを作成
            _files.Add(new FolderEntry(_directory) { Name = _definitions["name"] });
            // コレクションビューを作成
            _taskListView = new ListCollectionView(_taskList);
            _taskListView.SortDescriptions.Add(new SortDescription("FileName", ListSortDirection.Ascending));
            _taskListView.SortDescriptions.Add(new SortDescription("Line", ListSortDirection.Ascending));
            // タスクリストをバックグラウンドで作成
            _worker.DoWork += new DoWorkEventHandler(Worker_DoWork);
            _worker.RunWorkerCompleted += new RunWorkerCompletedEventHandler(Worker_RunWorkerCompleted);
            RefreshTaskList();
        }

        private string _directory;
        private readonly Dictionary<string, string> _definitions = new Dictionary<string,string>();

        private readonly BackgroundWorker _worker = new BackgroundWorker();

        private IDebuggable _module;

        private string _output;
        private CommandMode _mode;

        private readonly ObservableCollection<FileEntry> _files = new ObservableCollection<FileEntry>();

        private readonly ObservableCollection<Task> _taskList = new ObservableCollection<Task>();

        private readonly ListCollectionView _taskListView;

        private readonly ObservableCollection<Error> _errorList = new ObservableCollection<Error>();

        private readonly ObservableCollection<IEditableFile> _editingFiles = new ObservableCollection<IEditableFile>();

        private int historyIndex = -1;
        private readonly List<string> commandHistory = new List<string>();

        public int HistoryIndex
        {
            get { return historyIndex; }
            set { historyIndex = value; }
        }

        public List<string> CommandHistory
        {
            get { return commandHistory; }
        }

        public string RootDirectory
        {
            get { return _directory; }
        }

        public string Name
        {
            get { return _definitions["name"]; }
        }

        public IDebuggable Module
        {
            get { return _module; }
            private set
            {
                _module = value;
                OnPropertyChanged("Module");
            }
        }

        public bool IsDebugging
        {
            get { return _module != null; }
        }

        public string Output
        {
            get { return _output; }
            set
            {
                if (_output != value)
                {
                    _output = value;
                    OnPropertyChanged("Output");
                }
            }
        }

        public CommandMode Mode
        {
            get { return _mode; }
            set
            {
                if (_mode != value)
                {
                    _mode = value;
                    OnPropertyChanged("Mode");
                }
            }
        }

        public ObservableCollection<FileEntry> Files
        {
            get { return _files; }
        }

        public ObservableCollection<Task> TaskList
        {
            get { return _taskList; }
        }

        public ListCollectionView TaskListView
        {
            get { return _taskListView; }
        }

        public ObservableCollection<Error> ErrorList
        {
            get { return _errorList; }
        }

        public ObservableCollection<IEditableFile> EditingFiles
        {
            get { return _editingFiles; }
        }

        public IEditableFile AddDictionaryFile()
        {
            var item = new DictionaryFile { FileName = "無題" };
            _editingFiles.Add(item);
            return item;
        }

        public IEditableFile AddEditableFile(string path)
        {
            if (!File.Exists(path) || !DictionaryFile.IsEditableFile(path))
            {
                return null;
            }
            var item = _editingFiles.SingleOrDefault(p => p.FullPath == path);
            if (item == null)
            {
                item = new DictionaryFile(path);
                _editingFiles.Add(item);
            }
            return item;
        }

        public bool StartDebug()
        {
            if (_module != null)
            {
                return true;
            }
            try
            {
                Module = new LocalModule(_directory.EndsWith(@"\") ? _directory : _directory + @"\");
            }
            catch
            {
                // デバッグ不可能
                return false;
            }
            _module.Refresh();
            return true;
        }

        public void StopDebug()
        {
            if (_module != null)
            {
                _module.Dispose();
                Module = null;
            }
        }

        public bool RestartDebug()
        {
            StopDebug();
            return StartDebug();
        }

        public bool AttachDebug()
        {
            var owner = System.Windows.Application.Current.MainWindow;
            var dialog = new AttachDialog { Owner = owner };
            if (!(dialog.ShowDialog() ?? true))
            {
                return false;
            }
            var handle = dialog.SelectedTarget;
            try
            {
                Module = new RemoteModule(handle, owner);
            }
            catch
            {
                return false;
            }
            _module.Refresh();
            return true;
        }

        public void RefreshTaskList()
        {
            _taskList.Clear();
            _worker.RunWorkerAsync(Directory.GetFiles(_directory, "*.kis", SearchOption.AllDirectories));
        }

        public void RefreshTaskList(string path)
        {
            var removeItems = _taskList.Where(p => p.FullPath == path).ToList();
            foreach (var item in removeItems)
            {
                _taskList.Remove(item);
            }
            _worker.RunWorkerAsync(new[] { path });
        }

        private void Worker_DoWork(object sender, DoWorkEventArgs e)
        {
            var files = (IEnumerable<string>)e.Argument;
            e.Result = TaskListParser.ParseTasks(files);
        }

        private void Worker_RunWorkerCompleted(object sender, RunWorkerCompletedEventArgs e)
        {
            var list = (IEnumerable<Task>)e.Result;
            foreach (var item in list)
            {
                _taskList.Add(item);
            }
        }
    }
}
