﻿using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.Collections.Specialized;
using System.ComponentModel;
using System.IO;
using System.Linq;
using System.Text;
using System.Windows.Data;

namespace Hosui.Debugger
{
    public abstract class AbstractModule : IDebuggable
    {
        public AbstractModule()
        {
            _entries.CollectionChanged += new NotifyCollectionChangedEventHandler(Variables_CollectionChanged);
            _functions.CollectionChanged += new NotifyCollectionChangedEventHandler(Functions_CollectionChanged);
            _entriesView = new ListCollectionView(_entries);
            _entriesView.SortDescriptions.Add(new SortDescription("Name", ListSortDirection.Ascending));
            _functionsView = new ListCollectionView(_functions);
        }

        public abstract string Debug(string request);

        #region IDebuggable メンバ

        public string Evaluate(string script)
        {
            var result = Debug(ShioriVerbs.Get, "Evaluate", script);
            return result.Length > 0 ? result[0] : string.Empty;
        }

        public void Refresh()
        {
            RefreshVariables();
        }

        private ObservableCollection<Entry> _entries = new ObservableCollection<Entry>();

        public ObservableCollection<Entry> Entries
        {
            get { return _entries; }
        }

        private readonly ListCollectionView _entriesView;

        public CollectionView EntriesView
        {
            get { return _entriesView; }
        }

        private ObservableCollection<Function> _functions = new ObservableCollection<Function>();

        public ObservableCollection<Function> Functions
        {
            get { return _functions; }
        }

        private readonly ListCollectionView _functionsView;

        public CollectionView FunctionsView
        {
            get { return _functionsView; }
        }

        #endregion

        #region IDisposable メンバ

        public abstract void Dispose();

        #endregion

        private string[] Debug(ShioriVerbs verb, string id, params string[] param)
        {
            var request = new StringBuilder();
            switch (verb)
            {
                case ShioriVerbs.Get:
                    request.Append("GET");
                    break;
                case ShioriVerbs.Post:
                    request.Append("POST");
                    break;
                case ShioriVerbs.Put:
                    request.Append("PUT");
                    break;
                case ShioriVerbs.Delete:
                    request.Append("DELETE");
                    break;
                default:
                    return null;
            }
            request.AppendLine(" DEBUG/1.0");
            request.AppendFormat("ID: {0}\r\n", id);
            for (int i = 0; i < param.Length; i++)
            {
                request.AppendFormat("Reference{0}: {1}\r\n", i, param[i]);
            }
            request.AppendLine();
            var list = new SortedList<int, string>();
            var response = Debug(request.ToString());
            using (var reader = new StringReader(response))
            {
                reader.ReadLine();
                while (reader.Peek() != -1)
                {
                    var line = reader.ReadLine();
                    if (!line.StartsWith("Value"))
                    {
                        continue;
                    }
                    int index = line.IndexOf(':');
                    int order = Convert.ToInt32(line.Substring(5, index - 5));
                    list.Add(order, line.Substring(index + 1).Trim().Replace(@"\\", @"\"));
                }
            }
            return list.Select(p => p.Value).ToArray();
        }

        private bool _updatingVariableInternal = false;

        private void RefreshVariables()
        {
            var result = Debug(ShioriVerbs.Get, "Variables");
            // CollectionChanged を一時的に通知させない
            _updatingVariableInternal = true;
            // 全エントリを削除する
            _entries.Clear();
            // 変数リストを構築する
            foreach (var item in result)
            {
                Entry entry = null;
                Entry parent = null;
                var entries = _entries;
                int index = item.IndexOf('\x1');
                var name = item.Substring(0, index).Split('.');
                var values = item.Substring(index + 1).Split('\x2');
                for (int i = 0; i < name.Length; i++)
                {
                    entry = entries.FirstOrDefault(p => p.Name == name[i]);
                    if (entry == null)
                    {
                        entry = new Entry(name[i], parent);
                        entries.Add(entry);
                    }
                    parent = entry;
                    entries = entry.SubEntries;
                }
                if (entry != null)
                {
                    entry.Values = values;
                    entry.PropertyChanged += new PropertyChangedEventHandler(Variable_PropertyChanged);
                }
            }
            // CollectionChanged を有効にする
            _updatingVariableInternal = false;
        }

        private void CreateVariable(Entry entry)
        {
            var param = new List<string>();
            param.Add(entry.Name);
            param.AddRange(entry.Values);
            Debug(ShioriVerbs.Post, "Variable", param.ToArray());
            entry.PropertyChanged += new PropertyChangedEventHandler(Variable_PropertyChanged);
        }

        private void UpdateVariable(Entry variable)
        {
            var param = new List<string>();
            param.Add(variable.Name);
            param.AddRange(variable.Values);
            Debug(ShioriVerbs.Put, "Variable", param.ToArray());
        }

        private void DeleteVariable(Entry variable)
        {
            variable.PropertyChanged -= new PropertyChangedEventHandler(Variable_PropertyChanged);
            Debug(ShioriVerbs.Delete, "Variable", variable.Name);
        }

        private void Variables_CollectionChanged(object sender, NotifyCollectionChangedEventArgs e)
        {
            if (_updatingVariableInternal)
            {
                return;
            }
            switch (e.Action)
            {
                case NotifyCollectionChangedAction.Add:
                    foreach (Entry item in e.NewItems)
                    {
                        CreateVariable(item);
                    }
                    break;
                case NotifyCollectionChangedAction.Remove:
                    foreach (Entry item in e.OldItems)
                    {
                        DeleteVariable(item);
                    }
                    break;
                case NotifyCollectionChangedAction.Replace:
                case NotifyCollectionChangedAction.Reset:
                    foreach (Entry item in e.OldItems)
                    {
                        DeleteVariable(item);
                    }
                    foreach (Entry item in e.NewItems)
                    {
                        CreateVariable(item);
                    }
                    break;
                default:
                    return;
            }
        }

        private void Variable_PropertyChanged(object sender, PropertyChangedEventArgs e)
        {
            if (_updatingVariableInternal)
            {
                return;
            }
            UpdateVariable((Entry)sender);
        }

        private void Functions_CollectionChanged(object sender, NotifyCollectionChangedEventArgs e)
        {
            throw new NotImplementedException();
        }
    }
}
